# tools_info.py — Aggregator of productivity tools + Google OAuth + Perfex + ALL_TOOLS
import os
import re
import datetime as dt
from typing import Optional

import requests
from livekit.agents import function_tool, RunContext

# ---- Reuse core tools (already defined in tools.py) ----
from tools import (
    # core gmail + utilities
    gmail_read_by_uid,
    get_weather,
    send_email,
    gmail_smtp_healthcheck,
    gmail_search,
    gmail_reply_by_uid,
    trigger_n8n,
)

# ---- Perfex toolkit (full CRUD+search+summaries + line items) ----
from tools_perfex import (
    perfex_healthcheck,
    # Items
    perfex_get_item, perfex_search_items, perfex_search_items_summary,
    # Contacts
    perfex_add_contact, perfex_list_customer_contacts, perfex_search_contacts,
    perfex_search_contacts_summary, perfex_update_contact,
    # Customers
    perfex_add_customer, perfex_get_customer, perfex_search_customers,
    perfex_search_customers_summary, perfex_update_customer,
    # Estimates (+ line items)
    perfex_add_estimate, perfex_get_estimate, perfex_search_estimates,
    perfex_search_estimates_summary, perfex_update_estimate, perfex_add_estimate_item,
    # Expenses
    perfex_add_expense, perfex_get_expense, perfex_search_expenses,
    perfex_search_expenses_summary, perfex_update_expense,
    # Invoices (+ line items)
    perfex_add_invoice, perfex_get_invoice, perfex_search_invoices,
    perfex_search_invoices_summary, perfex_update_invoice, perfex_add_invoice_item,
    # Payments
    perfex_add_payment, perfex_list_payments, perfex_search_payments,
    perfex_search_payments_summary,
    # Projects
    perfex_add_project, perfex_get_project, perfex_search_projects,
    perfex_search_projects_summary, perfex_update_project,
    # Staffs
    perfex_add_staff, perfex_get_staff, perfex_search_staffs,
    perfex_search_staffs_summary, perfex_update_staff,
    # Tasks
    perfex_add_task, perfex_get_task, perfex_search_tasks,
    perfex_search_tasks_summary, perfex_update_task,
)

# ---- Google APIs (Calendar + Drive/Docs) ----
from google.oauth2.credentials import Credentials
from google_auth_oauthlib.flow import InstalledAppFlow
from google.auth.transport.requests import Request as GoogleRequest
from googleapiclient.discovery import build

SCOPES = [
    "https://www.googleapis.com/auth/calendar",
    "https://www.googleapis.com/auth/drive.readonly",
    "https://www.googleapis.com/auth/documents.readonly",
]
CRED_FILE = os.getenv("GOOGLE_CREDENTIALS_FILE", "credentials.json")
TOKEN_FILE = os.getenv("GOOGLE_TOKEN_FILE", "token.json")
USER_TZ = os.getenv("USER_TIMEZONE", "Africa/Accra")

def _google_creds() -> Credentials:
    creds = None
    if os.path.exists(TOKEN_FILE):
        creds = Credentials.from_authorized_user_file(TOKEN_FILE, SCOPES)
    if not creds or not creds.valid:
        if creds and creds.expired and creds.refresh_token:
            creds.refresh(GoogleRequest())
        else:
            if not os.path.exists(CRED_FILE):
                raise RuntimeError(
                    f"Missing {CRED_FILE}. Download your OAuth client (desktop) JSON "
                    "from Google Cloud Console and save it in your project root."
                )
            flow = InstalledAppFlow.from_client_secrets_file(CRED_FILE, SCOPES)
            creds = flow.run_local_server(port=0)
        with open(TOKEN_FILE, "w") as f:
            f.write(creds.to_json())
    return creds

# ----------------------------- Email summarization -----------------------------
@function_tool()
async def summarize_email(context: RunContext, uid: str) -> str:
    """
    Summarize a Gmail email body by IMAP UID (uses gmail_read_by_uid from tools.py).
    """
    info = await gmail_read_by_uid(context, uid)
    body = (info.get("body") or "").strip()
    if not body:
        return "No body to summarize."
    sentences = [s.strip() for s in body.replace("\n", " ").split(".") if s.strip()]
    points = sentences[:3]
    return "Summary: " + ". ".join(points) + ("." if points else "")

# ----------------------------- Google Calendar -----------------------------
@function_tool()
async def list_calendar_events(context: RunContext, max_results: int = 5) -> str:
    """
    List upcoming events from the primary Google Calendar.
    """
    creds = _google_creds()
    service = build("calendar", "v3", credentials=creds)
    now = dt.datetime.utcnow().isoformat() + "Z"
    events_result = (
        service.events()
        .list(
            calendarId="primary",
            timeMin=now,
            maxResults=max_results,
            singleEvents=True,
            orderBy="startTime",
            timeZone=USER_TZ,
        )
        .execute()
    )
    events = events_result.get("items", [])
    if not events:
        return "No upcoming events found."
    def _fmt(ev):
        start = ev["start"].get("dateTime") or ev["start"].get("date")
        end = ev["end"].get("dateTime") or ev["end"].get("date")
        title = ev.get("summary", "(no title)")
        loc = ev.get("location", "")
        return f"{start} → {end} — {title}" + (f" @ {loc}" if loc else "")
    return "\n".join(_fmt(ev) for ev in events)

@function_tool()
async def add_calendar_event(
    context: RunContext,
    title: str,
    date: str,               # "YYYY-MM-DD"
    time: str,               # "HH:MM" 24h
    duration_minutes: int = 60,
    description: str = "",
    location: Optional[str] = None,
) -> str:
    """
    Create a timed event on the primary Google Calendar.
    """
    start_local = dt.datetime.strptime(f"{date} {time}", "%Y-%m-%d %H:%M")
    end_local = start_local + dt.timedelta(minutes=duration_minutes)
    creds = _google_creds()
    service = build("calendar", "v3", credentials=creds)
    event_body = {
        "summary": title,
        "description": description,
        "start": {"dateTime": start_local.isoformat(), "timeZone": USER_TZ},
        "end": {"dateTime": end_local.isoformat(), "timeZone": USER_TZ},
    }
    if location:
        event_body["location"] = location
    created = service.events().insert(calendarId="primary", body=event_body).execute()
    link = created.get("htmlLink", "(no link)")
    return f"Event created: {title} on {date} {time} ({duration_minutes}m). Link: {link}"

# ----------------------------- Drive/Docs Summarization -----------------------------
_DRIVE_FILE_ID_RE = re.compile(r"/d/([a-zA-Z0-9_-]+)")

def _extract_drive_file_id(url_or_id: str) -> str:
    m = _DRIVE_FILE_ID_RE.search(url_or_id)
    return m.group(1) if m else url_or_id

@function_tool()
async def summarize_document(context: RunContext, file_url_or_id: str) -> str:
    """
    Summarize a Google Doc or a Drive file (tries Docs API; otherwise Drive export to text/plain).
    """
    creds = _google_creds()
    doc_service = build("docs", "v1", credentials=creds)
    drive_service = build("drive", "v3", credentials=creds)
    file_id = _extract_drive_file_id(file_url_or_id)
    meta = drive_service.files().get(fileId=file_id, fields="id,name,mimeType").execute()
    mime = meta.get("mimeType", "")
    name = meta.get("name", file_id)

    def _summ(text: str) -> str:
        text = " ".join(text.split())
        if not text:
            return f"{name}: (no extractable text)"
        return f"{name} — Summary: {text[:500]}{'...' if len(text)>500 else ''}"

    try:
        if mime == "application/vnd.google-apps.document":
            doc = doc_service.documents().get(documentId=file_id).execute()
            buf = []
            for el in doc.get("body", {}).get("content", []):
                p = el.get("paragraph")
                if not p: continue
                for e in p.get("elements", []):
                    buf.append(e.get("textRun", {}).get("content", ""))
            return _summ("".join(buf))
        else:
            resp = drive_service.files().export(fileId=file_id, mimeType="text/plain").execute()
            txt = resp.decode("utf-8", errors="replace") if isinstance(resp, bytes) else str(resp)
            if not txt.strip():
                return f"{name}: Exported text is empty."
            return _summ(txt)
    except Exception as e:
        return f"Failed to summarize '{name}': {e}"

# ----------------------------- Meeting transcript summarization -----------------------------
@function_tool()
async def summarize_meeting(context: RunContext, transcript_url: str) -> str:
    try:
        r = requests.get(transcript_url, timeout=20)
        lines = r.text.splitlines()
        return "Meeting summary: " + " ".join(lines[:5])
    except Exception as e:
        return f"Failed to fetch transcript: {e}"

# ----------------------------- Task Router -----------------------------
@function_tool()
async def create_task(
    context: RunContext,
    service: str,
    title: str,
    description: str = "",
    startdate: Optional[str] = None,
    duedate: Optional[str] = None,
    priority: Optional[str] = None,
    rel_type: Optional[str] = None,
    rel_id: Optional[int] = None,
) -> str:
    """
    Router for task creation. Currently supports 'perfex'.
    """
    svc = (service or "").strip().lower()
    if svc == "perfex":
        return await perfex_add_task(context, title, description, startdate, duedate, priority, rel_type, rel_id)
    return f"Stub: would create task '{title}' in {svc or 'unknown service'}; only 'perfex' is implemented now."

# ----------------------------- ALL_TOOLS -----------------------------
# Expose everything in one place so agent.py can register with tools=ALL_TOOLS
ALL_TOOLS = [
    # Core tools (from tools.py)
    get_weather, send_email, gmail_smtp_healthcheck,
    gmail_search, gmail_read_by_uid, gmail_reply_by_uid,
    trigger_n8n,

    # Productivity / info
    summarize_email, list_calendar_events, add_calendar_event,
    summarize_document, summarize_meeting, create_task,

    # Perfex: healthcheck
    perfex_healthcheck,

    # Perfex: Items
    perfex_get_item, perfex_search_items, perfex_search_items_summary,

    # Perfex: Contacts
    perfex_add_contact, perfex_list_customer_contacts, perfex_search_contacts,
    perfex_search_contacts_summary, perfex_update_contact,

    # Perfex: Customers
    perfex_add_customer, perfex_get_customer, perfex_search_customers,
    perfex_search_customers_summary, perfex_update_customer,

    # Perfex: Estimates (+ items)
    perfex_add_estimate, perfex_get_estimate, perfex_search_estimates,
    perfex_search_estimates_summary, perfex_update_estimate, perfex_add_estimate_item,

    # Perfex: Expenses
    perfex_add_expense, perfex_get_expense, perfex_search_expenses,
    perfex_search_expenses_summary, perfex_update_expense,

    # Perfex: Invoices (+ items)
    perfex_add_invoice, perfex_get_invoice, perfex_search_invoices,
    perfex_search_invoices_summary, perfex_update_invoice, perfex_add_invoice_item,

    # Perfex: Payments
    perfex_add_payment, perfex_list_payments, perfex_search_payments,
    perfex_search_payments_summary,

    # Perfex: Projects
    perfex_add_project, perfex_get_project, perfex_search_projects,
    perfex_search_projects_summary, perfex_update_project,

    # Perfex: Staffs
    perfex_add_staff, perfex_get_staff, perfex_search_staffs,
    perfex_search_staffs_summary, perfex_update_staff,

    # Perfex: Tasks
    perfex_add_task, perfex_get_task, perfex_search_tasks,
    perfex_search_tasks_summary, perfex_update_task,
]