# tools_perfex.py — Perfex CRM tools (CRUD + search summaries + line items; no deletes)
import os
import requests
from typing import Any, Optional, Dict, List
from livekit.agents import function_tool, RunContext

# ------------------------------------------------------------------------------
# CONFIG
#   PERFEX_API_URL should be the FULL API base, e.g.:
#     https://office.example.com/admin/api
#   DO NOT end it with a trailing slash.
#   PERFEX_API_KEY is the authtoken value from Perfex API management.
# ------------------------------------------------------------------------------

def _perfex_base_url(endpoint: str) -> str:
    base = os.getenv("PERFEX_API_URL", "").rstrip("/")
    if not base:
        raise RuntimeError("PERFEX_API_URL is not set")
    return f"{base}/{endpoint.lstrip('/')}"

def _perfex_headers() -> Dict[str, str]:
    token = os.getenv("PERFEX_API_KEY", "")
    if not token:
        raise RuntimeError("PERFEX_API_KEY is not set")
    return {"authtoken": token}

def _json_or_text_any(r: requests.Response) -> Any:
    try:
        return r.json()
    except Exception:
        return {"text": r.text}

def _resp_ok_text(r: requests.Response, default_ok_msg: str = "OK") -> str:
    j = _json_or_text_any(r)
    status = j.get("status")
    msg = j.get("message") or j.get("text", "")[:1000]
    if r.ok and (status is True or status == "true"):
        rid = j.get("id") or j.get("data", {}).get("id") or ""
        return f"Success: {default_ok_msg}" + (f" (id={rid})" if rid else f" — {msg}")
    return f"Failed ({r.status_code}): {msg}"

def _get(endpoint: str, params: Optional[Dict[str, Any]] = None) -> requests.Response:
    url = _perfex_base_url(endpoint)
    headers = _perfex_headers()
    return requests.get(url, headers=headers, params=params or {}, timeout=30)

def _post(endpoint: str, data: Dict[str, Any]) -> requests.Response:
    url = _perfex_base_url(endpoint)
    headers = _perfex_headers()
    return requests.post(url, headers=headers, data=data, timeout=30)

def _put(endpoint: str, data: Dict[str, Any]) -> requests.Response:
    url = _perfex_base_url(endpoint)
    headers = _perfex_headers()
    return requests.put(url, headers=headers, data=data, timeout=30)

# ------------------------------------------------------------------------------
# HEALTHCHECK
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_healthcheck(context: RunContext) -> str:
    """Quick GET /clients to verify token & base URL."""
    try:
        r = _get("/clients")
        if not r.ok:
            return _resp_ok_text(r, default_ok_msg="healthcheck")
        j = _json_or_text_any(r)
        clients = j.get("clients") or j.get("data") or []
        return f"Healthcheck OK — clients: {len(clients)}"
    except Exception as e:
        return f"Healthcheck exception: {e}"

# ------------------------------------------------------------------------------
# UTILS: compact summaries
# ------------------------------------------------------------------------------

def _list_summary(rows: List[Dict[str, Any]], fields: List[str], id_keys: List[str]) -> str:
    out = []
    for row in rows:
        if not isinstance(row, dict):
            out.append(f"- {str(row)[:200]}")
            continue
        rid = next((str(row.get(k)) for k in id_keys if row.get(k) is not None), "?")
        parts = []
        for f in fields:
            val = row.get(f)
            if val not in (None, ""):
                parts.append(f"{f}={val}")
        label = ", ".join(parts) if parts else "(no fields)"
        out.append(f"- id={rid}: {label}")
    return "\n".join(out) if out else "(no results)"

def _summary_from_response(
    r: requests.Response,
    data_keys: List[str],
    fields: List[str],
    id_keys: List[str]
) -> str:
    if not r.ok:
        # reuse existing formatter if you have it
        return _resp_ok_text(r, default_ok_msg="fetch")

    j = _json_or_text_any(r)

    # Case 1: top-level list
    if isinstance(j, list):
        return _list_summary(j, fields=fields, id_keys=id_keys)

    # Case 2: top-level dict; try known array keys or 'data'
    if isinstance(j, dict):
        rows = None
        for k in data_keys:
            v = j.get(k)
            if isinstance(v, list):
                rows = v
                break
        if rows is None and isinstance(j.get("data"), list):
            rows = j.get("data")

        if isinstance(rows, list):
            return _list_summary(rows, fields=fields, id_keys=id_keys)

        # Fallback: show trimmed JSON if no list found
        return str(j)[:1200]

    # Unknown shape; return a trimmed string
    return str(j)[:1200]

# ------------------------------------------------------------------------------
# ITEMS
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_get_item(context: RunContext, item_id: str) -> str:
    r = _get(f"/items/{item_id}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="fetch item")

@function_tool()
async def perfex_search_items(context: RunContext, keysearch: str) -> str:
    r = _get(f"/items/search/{keysearch}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="search items")

@function_tool()
async def perfex_search_items_summary(context: RunContext, keysearch: str) -> str:
    r = _get(f"/items/search/{keysearch}")
    return _summary_from_response(r, data_keys=["items"], fields=["description", "rate"], id_keys=["id","itemid"])

# ------------------------------------------------------------------------------
# CONTACTS
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_add_contact(
    context: RunContext,
    firstname: str,
    lastname: str,
    email: str,
    phonenumber: Optional[str] = None,
    customer_id: Optional[str] = None,
) -> str:
    data = {"firstname": firstname, "lastname": lastname, "email": email}
    if phonenumber: data["phonenumber"] = phonenumber
    if customer_id: data["userid"] = customer_id
    r = _post("/contacts", data)
    return _resp_ok_text(r, default_ok_msg="contact created")

@function_tool()
async def perfex_list_customer_contacts(context: RunContext, customer_id: str, contact_id: Optional[str] = None) -> str:
    endpoint = f"/contacts/{customer_id}"
    if contact_id:
        endpoint += f"/{contact_id}"
    r = _get(endpoint)
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="list contacts")

@function_tool()
async def perfex_search_contacts(context: RunContext, keysearch: str) -> str:
    r = _get(f"/contacts/search/{keysearch}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="search contacts")

@function_tool()
async def perfex_search_contacts_summary(context: RunContext, keysearch: str) -> str:
    r = _get(f"/contacts/search/{keysearch}")
    return _summary_from_response(r, data_keys=["contacts"], fields=["firstname","lastname","email","phonenumber"], id_keys=["id","userid"])

@function_tool()
async def perfex_update_contact(
    context: RunContext,
    contact_id: str,
    firstname: Optional[str] = None,
    lastname: Optional[str] = None,
    email: Optional[str] = None,
    phonenumber: Optional[str] = None,
    active: Optional[str] = None,  # "0"/"1"
) -> str:
    data: Dict[str, Any] = {}
    if firstname is not None: data["firstname"] = firstname
    if lastname  is not None: data["lastname"]  = lastname
    if email     is not None: data["email"]     = email
    if phonenumber is not None: data["phonenumber"] = phonenumber
    if active is not None: data["active"] = active
    r = _put(f"/contacts/{contact_id}", data)
    return _resp_ok_text(r, default_ok_msg="contact updated")

# ------------------------------------------------------------------------------
# CUSTOMERS
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_add_customer(
    context: RunContext,
    company: str,
    vat: Optional[str] = None,
    phonenumber: Optional[str] = None,
    website: Optional[str] = None,
    address: Optional[str] = None,
    city: Optional[str] = None,
    country: Optional[str] = None,
) -> str:
    data = {"company": company}
    if vat: data["vat"] = vat
    if phonenumber: data["phonenumber"] = phonenumber
    if website: data["website"] = website
    if address: data["address"] = address
    if city: data["city"] = city
    if country: data["country"] = country
    r = _post("/customers", data)
    return _resp_ok_text(r, default_ok_msg="customer created")

@function_tool()
async def perfex_get_customer(context: RunContext, customer_id: str) -> str:
    r = _get(f"/customers/{customer_id}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="fetch customer")

@function_tool()
async def perfex_search_customers(context: RunContext, keysearch: str) -> str:
    r = _get(f"/customers/search/{keysearch}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="search customers")

@function_tool()
async def perfex_search_customers_summary(context: RunContext, keysearch: str) -> str:
    r = _get(f"/customers/search/{keysearch}")
    return _summary_from_response(r, data_keys=["customers"], fields=["company","phonenumber","city","country"], id_keys=["id","userid"])

@function_tool()
async def perfex_update_customer(
    context: RunContext,
    customer_id: str,
    company: Optional[str] = None,
    vat: Optional[str] = None,
    phonenumber: Optional[str] = None,
    website: Optional[str] = None,
    address: Optional[str] = None,
    city: Optional[str] = None,
    country: Optional[str] = None,
    active: Optional[str] = None,  # "0"/"1"
) -> str:
    data: Dict[str, Any] = {}
    if company: data["company"] = company
    if vat: data["vat"] = vat
    if phonenumber: data["phonenumber"] = phonenumber
    if website: data["website"] = website
    if address: data["address"] = address
    if city: data["city"] = city
    if country: data["country"] = country
    if active is not None: data["active"] = active
    r = _put(f"/customers/{customer_id}", data)
    return _resp_ok_text(r, default_ok_msg="customer updated")

# ------------------------------------------------------------------------------
# ESTIMATES (incl. line items)
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_add_estimate(
    context: RunContext,
    customer_id: str,
    date: str,            # YYYY-MM-DD
    expirydate: str,      # YYYY-MM-DD
    subtotal: float,
    total: float,
    description: str = "",
) -> str:
    data = {
        "clientid": customer_id,
        "date": date,
        "expirydate": expirydate,
        "subtotal": str(subtotal),
        "total": str(total),
        "adminnote": description,
    }
    r = _post("/estimates", data)
    return _resp_ok_text(r, default_ok_msg="estimate created")

@function_tool()
async def perfex_get_estimate(context: RunContext, estimate_id: str) -> str:
    r = _get(f"/estimates/{estimate_id}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="fetch estimate")

@function_tool()
async def perfex_search_estimates(context: RunContext, keysearch: str) -> str:
    r = _get(f"/estimates/search/{keysearch}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="search estimates")

@function_tool()
async def perfex_search_estimates_summary(context: RunContext, keysearch: str) -> str:
    r = _get(f"/estimates/search/{keysearch}")
    return _summary_from_response(r, data_keys=["estimates"], fields=["clientid","date","expirydate","total"], id_keys=["id","estimateid"])

@function_tool()
async def perfex_update_estimate(
    context: RunContext,
    estimate_id: str,
    date: Optional[str] = None,
    expirydate: Optional[str] = None,
    subtotal: Optional[float] = None,
    total: Optional[float] = None,
    description: Optional[str] = None,
) -> str:
    data: Dict[str, Any] = {}
    if date: data["date"] = date
    if expirydate: data["expirydate"] = expirydate
    if subtotal is not None: data["subtotal"] = str(subtotal)
    if total is not None: data["total"] = str(total)
    if description is not None: data["adminnote"] = description
    r = _put(f"/estimates/{estimate_id}", data)
    return _resp_ok_text(r, default_ok_msg="estimate updated")

@function_tool()
async def perfex_add_estimate_item(
    context: RunContext,
    estimate_id: str,
    description: str,
    qty: float,
    rate: float,
    long_description: Optional[str] = None,
    taxid: Optional[str] = None,
    itemid: Optional[str] = None,   # reference existing item
    unit: Optional[str] = None,
) -> str:
    """
    Add a line item to an estimate.
    Common Perfex REST patterns: POST /estimates/:id/items with form-data.
    """
    data: Dict[str, Any] = {
        "description": description,
        "qty": str(qty),
        "rate": str(rate),
    }
    if long_description: data["long_description"] = long_description
    if taxid: data["taxid"] = taxid
    if itemid: data["itemid"] = itemid
    if unit: data["unit"] = unit
    r = _post(f"/estimates/{estimate_id}/items", data)
    return _resp_ok_text(r, default_ok_msg="estimate item added")

# ------------------------------------------------------------------------------
# EXPENSES
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_add_expense(
    context: RunContext,
    category_id: str,     # numeric category id
    amount: float,
    date: str,            # YYYY-MM-DD
    note: str = "",
) -> str:
    data = {"category": category_id, "amount": str(amount), "date": date, "note": note}
    r = _post("/expenses", data)
    return _resp_ok_text(r, default_ok_msg="expense created")

@function_tool()
async def perfex_get_expense(context: RunContext, expense_id: str) -> str:
    r = _get(f"/expenses/{expense_id}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="fetch expense")

@function_tool()
async def perfex_search_expenses(context: RunContext, keysearch: str) -> str:
    r = _get(f"/expenses/search/{keysearch}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="search expenses")

@function_tool()
async def perfex_search_expenses_summary(context: RunContext, keysearch: str) -> str:
    r = _get(f"/expenses/search/{keysearch}")
    return _summary_from_response(r, data_keys=["expenses"], fields=["category","amount","date","note"], id_keys=["id","expenseid"])

@function_tool()
async def perfex_update_expense(
    context: RunContext,
    expense_id: str,
    category_id: Optional[str] = None,
    amount: Optional[float] = None,
    date: Optional[str] = None,
    note: Optional[str] = None,
) -> str:
    # Your doc shows PUT /expenses (id in payload). If your build requires /expenses/:id, change here.
    data: Dict[str, Any] = {"id": expense_id}
    if category_id is not None: data["category"] = category_id
    if amount is not None: data["amount"] = str(amount)
    if date is not None: data["date"] = date
    if note is not None: data["note"] = note
    r = _put("/expenses", data)
    return _resp_ok_text(r, default_ok_msg="expense updated")

# ------------------------------------------------------------------------------
# INVOICES (incl. line items)
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_add_invoice(
    context: RunContext,
    customer_id: str,
    date: str,            # YYYY-MM-DD
    duedate: str,         # YYYY-MM-DD
    subtotal: float,
    total: float,
    description: str = "",
) -> str:
    data = {
        "clientid": customer_id,
        "date": date,
        "duedate": duedate,
        "subtotal": str(subtotal),
        "total": str(total),
        "adminnote": description,
    }
    r = _post("/invoices", data)
    return _resp_ok_text(r, default_ok_msg="invoice created")

@function_tool()
async def perfex_get_invoice(context: RunContext, invoice_id: str) -> str:
    r = _get(f"/invoices/{invoice_id}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="fetch invoice")

@function_tool()
async def perfex_search_invoices(context: RunContext, keysearch: str) -> str:
    r = _get(f"/invoices/search/{keysearch}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="search invoices")

@function_tool()
async def perfex_search_invoices_summary(context: RunContext, keysearch: str) -> str:
    r = _get(f"/invoices/search/{keysearch}")
    return _summary_from_response(r, data_keys=["invoices"], fields=["clientid","date","duedate","total"], id_keys=["id","invoiceid"])

@function_tool()
async def perfex_update_invoice(
    context: RunContext,
    invoice_id: str,
    date: Optional[str] = None,
    duedate: Optional[str] = None,
    subtotal: Optional[float] = None,
    total: Optional[float] = None,
    description: Optional[str] = None,
) -> str:
    data: Dict[str, Any] = {}
    if date: data["date"] = date
    if duedate: data["duedate"] = duedate
    if subtotal is not None: data["subtotal"] = str(subtotal)
    if total is not None: data["total"] = str(total)
    if description is not None: data["adminnote"] = description
    r = _put(f"/invoices/{invoice_id}", data)
    return _resp_ok_text(r, default_ok_msg="invoice updated")

@function_tool()
async def perfex_add_invoice_item(
    context: RunContext,
    invoice_id: str,
    description: str,
    qty: float,
    rate: float,
    long_description: Optional[str] = None,
    taxid: Optional[str] = None,
    itemid: Optional[str] = None,   # reference existing item
    unit: Optional[str] = None,
) -> str:
    """
    Add a line item to an invoice.
    Common Perfex REST patterns: POST /invoices/:id/items with form-data.
    """
    data: Dict[str, Any] = {
        "description": description,
        "qty": str(qty),
        "rate": str(rate),
    }
    if long_description: data["long_description"] = long_description
    if taxid: data["taxid"] = taxid
    if itemid: data["itemid"] = itemid
    if unit: data["unit"] = unit
    r = _post(f"/invoices/{invoice_id}/items", data)
    return _resp_ok_text(r, default_ok_msg="invoice item added")

# ------------------------------------------------------------------------------
# PAYMENTS
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_add_payment(
    context: RunContext,
    amount: float,
    date: str,              # YYYY-MM-DD
    invoice_id: Optional[str] = None,
    customer_id: Optional[str] = None,
    payment_mode: Optional[str] = None,
    note: str = "",
) -> str:
    data: Dict[str, Any] = {"amount": str(amount), "date": date, "note": note}
    if invoice_id:  data["invoice"]  = invoice_id
    if customer_id: data["clientid"] = customer_id
    if payment_mode: data["paymentmode"] = payment_mode
    r = _post("/payments", data)
    return _resp_ok_text(r, default_ok_msg="payment created")

@function_tool()
async def perfex_list_payments(context: RunContext, id_or_limit: str) -> str:
    r = _get(f"/payments/{id_or_limit}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="list payments")

@function_tool()
async def perfex_search_payments(context: RunContext, keysearch: str) -> str:
    r = _get(f"/payments/search/{keysearch}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="search payments")

@function_tool()
async def perfex_search_payments_summary(context: RunContext, keysearch: str) -> str:
    r = _get(f"/payments/search/{keysearch}")
    return _summary_from_response(r, data_keys=["payments"], fields=["amount","date","invoice","clientid"], id_keys=["id","paymentid"])

# ------------------------------------------------------------------------------
# PROJECTS
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_add_project(
    context: RunContext,
    name: str,
    customer_id: Optional[str] = None,
    start_date: Optional[str] = None,
    deadline: Optional[str] = None,
    description: str = "",
) -> str:
    data: Dict[str, Any] = {"name": name, "description": description or ""}
    if customer_id: data["clientid"] = customer_id
    if start_date:  data["start_date"] = start_date
    if deadline:    data["deadline"] = deadline
    r = _post("/projects", data)
    return _resp_ok_text(r, default_ok_msg="project created")

@function_tool()
async def perfex_get_project(context: RunContext, project_id: str) -> str:
    r = _get(f"/projects/{project_id}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="fetch project")

@function_tool()
async def perfex_search_projects(context: RunContext, keysearch: str) -> str:
    r = _get(f"/projects/search/{keysearch}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="search projects")

@function_tool()
async def perfex_search_projects_summary(context: RunContext, keysearch: str) -> str:
    r = _get(f"/projects/search/{keysearch}")
    return _summary_from_response(r, data_keys=["projects"], fields=["name","clientid","start_date","deadline","status"], id_keys=["id","projectid"])

@function_tool()
async def perfex_update_project(
    context: RunContext,
    project_id: str,
    name: Optional[str] = None,
    start_date: Optional[str] = None,
    deadline: Optional[str] = None,
    description: Optional[str] = None,
    status: Optional[str] = None,
) -> str:
    data: Dict[str, Any] = {}
    if name is not None: data["name"] = name
    if start_date is not None: data["start_date"] = start_date
    if deadline is not None: data["deadline"] = deadline
    if description is not None: data["description"] = description
    if status is not None: data["status"] = status
    r = _put(f"/projects/{project_id}", data)
    return _resp_ok_text(r, default_ok_msg="project updated")

# ------------------------------------------------------------------------------
# STAFFS
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_add_staff(
    context: RunContext,
    firstname: str,
    lastname: str,
    email: str,
    phonenumber: Optional[str] = None,
    password: Optional[str] = None,
) -> str:
    data: Dict[str, Any] = {"firstname": firstname, "lastname": lastname, "email": email}
    if phonenumber: data["phonenumber"] = phonenumber
    if password: data["password"] = password
    r = _post("/staffs", data)
    return _resp_ok_text(r, default_ok_msg="staff created")

@function_tool()
async def perfex_get_staff(context: RunContext, staff_id: str) -> str:
    r = _get(f"/staffs/{staff_id}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="fetch staff")

@function_tool()
async def perfex_search_staffs(context: RunContext, keysearch: str) -> str:
    r = _get(f"/staffs/search/{keysearch}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="search staffs")

@function_tool()
async def perfex_search_staffs_summary(context: RunContext, keysearch: str) -> str:
    r = _get(f"/staffs/search/{keysearch}")
    return _summary_from_response(r, data_keys=["staffs"], fields=["firstname","lastname","email","phonenumber","active"], id_keys=["id","staffid"])

@function_tool()
async def perfex_update_staff(
    context: RunContext,
    staff_id: str,
    firstname: Optional[str] = None,
    lastname: Optional[str] = None,
    email: Optional[str] = None,
    phonenumber: Optional[str] = None,
    active: Optional[str] = None,
) -> str:
    data: Dict[str, Any] = {}
    if firstname is not None: data["firstname"] = firstname
    if lastname  is not None: data["lastname"]  = lastname
    if email     is not None: data["email"]     = email
    if phonenumber is not None: data["phonenumber"] = phonenumber
    if active is not None: data["active"] = active
    r = _put(f"/staffs/{staff_id}", data)
    return _resp_ok_text(r, default_ok_msg="staff updated")

# ------------------------------------------------------------------------------
# TASKS
# ------------------------------------------------------------------------------

@function_tool()
async def perfex_add_task(
    context: RunContext,
    title: str,
    description: str = "",
    startdate: Optional[str] = None,  # YYYY-MM-DD (often required)
    duedate: Optional[str] = None,
    priority: Optional[str] = None,
    rel_type: Optional[str] = None,
    rel_id: Optional[int] = None,
) -> str:
    if not title:
        return "Task 'title' is required."
    if not startdate:
        return "Task 'startdate' (YYYY-MM-DD) is required."
    data: Dict[str, Any] = {"name": title, "description": description or "", "startdate": startdate}
    if duedate: data["duedate"] = duedate
    if priority: data["priority"] = priority
    if rel_type: data["rel_type"] = rel_type
    if rel_id is not None: data["rel_id"] = str(rel_id)
    r = _post("/tasks", data)
    return _resp_ok_text(r, default_ok_msg="task created")

@function_tool()
async def perfex_get_task(context: RunContext, task_id: str) -> str:
    r = _get(f"/tasks/{task_id}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="fetch task")

@function_tool()
async def perfex_search_tasks(context: RunContext, keysearch: str) -> str:
    r = _get(f"/tasks/search/{keysearch}")
    return r.text[:4000] if r.ok else _resp_ok_text(r, default_ok_msg="search tasks")

@function_tool()
async def perfex_search_tasks_summary(context: RunContext, keysearch: str) -> str:
    r = _get(f"/tasks/search/{keysearch}")
    return _summary_from_response(r, data_keys=["tasks"], fields=["name","startdate","duedate","priority","status"], id_keys=["id","taskid"])

@function_tool()
async def perfex_update_task(
    context: RunContext,
    task_id: str,
    title: Optional[str] = None,
    description: Optional[str] = None,
    startdate: Optional[str] = None,
    duedate: Optional[str] = None,
    priority: Optional[str] = None,
    rel_type: Optional[str] = None,
    rel_id: Optional[int] = None,
    status: Optional[str] = None,
) -> str:
    data: Dict[str, Any] = {}
    if title is not None: data["name"] = title
    if description is not None: data["description"] = description
    if startdate is not None: data["startdate"] = startdate
    if duedate is not None: data["duedate"] = duedate
    if priority is not None: data["priority"] = priority
    if rel_type is not None: data["rel_type"] = rel_type
    if rel_id is not None: data["rel_id"] = str(rel_id)
    if status is not None: data["status"] = status
    r = _put(f"/tasks/{task_id}", data)
    return _resp_ok_text(r, default_ok_msg="task updated")